<?php
/*
Plugin Name: Tic Tac Toe Game
Description: Adds a simple tic tac toe game via shortcode [tic_tac_toe]
Version: 1.7
Author: Thomas Jean Nielsen
*/

if (!defined('ABSPATH')) exit;

// Register settings page
function ttt_add_settings_page() {
    add_options_page(
        'Tic Tac Toe Settings',
        'Tic Tac Toe',
        'manage_options',
        'ttt-settings',
        'ttt_render_settings_page'
    );
}
add_action('admin_menu', 'ttt_add_settings_page');

function ttt_render_settings_page() {
    ?>
    <div class="wrap">
    <h1>Tic Tac Toe Settings</h1>
    <p><strong>Use this shortcode [tic_tac_toe] on the page/post where you want the game.</strong></p>
        <form method="post" action="options.php">
            <?php
            settings_fields('ttt_settings_group');
            do_settings_sections('ttt-settings');
            submit_button();
            ?>
        </form>
    </div>
    <?php
}

function ttt_register_settings() {
    register_setting('ttt_settings_group', 'ttt_start_player');
    register_setting('ttt_settings_group', 'ttt_win_color');
    register_setting('ttt_settings_group', 'ttt_language');
    register_setting('ttt_settings_group', 'ttt_board_size');

    add_settings_section('ttt_main_section', 'General Settings', null, 'ttt-settings');

    add_settings_field('ttt_start_player', 'Starting Player', 'ttt_start_player_callback', 'ttt-settings', 'ttt_main_section');
    add_settings_field('ttt_win_color', 'Winning Line Color', 'ttt_win_color_callback', 'ttt-settings', 'ttt_main_section');
    add_settings_field('ttt_language', 'Language', 'ttt_language_callback', 'ttt-settings', 'ttt_main_section');
    add_settings_field('ttt_board_size', 'Board Size', 'ttt_board_size_callback', 'ttt-settings', 'ttt_main_section');
}
add_action('admin_init', 'ttt_register_settings');

function ttt_start_player_callback() {
    $value = get_option('ttt_start_player', 'X');
    ?>
    <select name="ttt_start_player">
        <option value="X" <?php selected($value, 'X'); ?>>X</option>
        <option value="O" <?php selected($value, 'O'); ?>>O</option>
        <option value="random" <?php selected($value, 'random'); ?>>Random</option>
    </select>
    <?php
}

function ttt_win_color_callback() {
    $value = get_option('ttt_win_color', 'green');
    ?>
    <select name="ttt_win_color">
        <option value="green" <?php selected($value, 'green'); ?>>Green</option>
        <option value="red" <?php selected($value, 'red'); ?>>Red</option>
        <option value="blue" <?php selected($value, 'blue'); ?>>Blue</option>
        <option value="orange" <?php selected($value, 'orange'); ?>>Orange</option>
    </select>
    <?php
}

function ttt_language_callback() {
    $value = get_option('ttt_language', 'en');
    ?>
    <select name="ttt_language">
        <option value="en" <?php selected($value, 'en'); ?>>English</option>
        <option value="da" <?php selected($value, 'da'); ?>>Dansk</option>
    </select>
    <?php
}

function ttt_board_size_callback() {
    $value = get_option('ttt_board_size', 'small');
    ?>
    <select name="ttt_board_size">
        <option value="small" <?php selected($value, 'small'); ?>>Small (180px)</option>
        <option value="medium" <?php selected($value, 'medium'); ?>>Medium (300px)</option>
        <option value="large" <?php selected($value, 'large'); ?>>Large (600px)</option>
    </select>
    <?php
}

function kjb_vis_spil() {
    $start_player = get_option('ttt_start_player', 'X');
    $win_color = get_option('ttt_win_color', 'green');
    $language = get_option('ttt_language', 'en');
    $board_size = get_option('ttt_board_size', 'small');

    $size_css = [
        'small' => 'width:180px;height:180px;',
        'medium' => 'width:300px;height:300px;',
        'large' => 'width:600px;height:600px;'
    ];

    $texts = [
        'en' => ['turn' => 'Turn: ', 'reset' => 'Reset Game', 'draw' => 'Draw!', 'wins' => ' wins!'],
        'da' => ['turn' => 'Tur: ', 'reset' => 'Nulstil spil', 'draw' => 'Uafgjort!', 'wins' => ' har vundet!']
    ];

    ob_start();
    ?>
    <div id="krydsbolle-spil">
        <style>
            #krydsbolle-spil table {
                border-collapse: collapse;
                margin: 10px auto;
                <?php echo $size_css[$board_size]; ?>
            }
            #krydsbolle-spil td {
                width: 33.33%;
                height: 33.33%;
                text-align: center;
                vertical-align: middle;
                font-size: <?php echo ($board_size === 'large') ? '4em' : (($board_size === 'medium') ? '3em' : '2em'); ?>;
                border: 2px solid #333;
                cursor: pointer;
            }
            #krydsbolle-spil td.vinder {
                color: <?php echo esc_attr($win_color); ?>;
                font-weight: bold;
            }
            #krydsbolle-status {
                text-align: center;
                font-weight: bold;
                margin-top: 10px;
            }
            #krydsbolle-reset {
                display: block;
                margin: 10px auto;
                padding: 5px 15px;
                font-size: 1em;
                cursor: pointer;
            }
        </style>
        <table>
            <tr>
                <td data-cell="0"></td>
                <td data-cell="1"></td>
                <td data-cell="2"></td>
            </tr>
            <tr>
                <td data-cell="3"></td>
                <td data-cell="4"></td>
                <td data-cell="5"></td>
            </tr>
            <tr>
                <td data-cell="6"></td>
                <td data-cell="7"></td>
                <td data-cell="8"></td>
            </tr>
        </table>
        <div id="krydsbolle-status"></div>
        <button id="krydsbolle-reset"><?php echo $texts[$language]['reset']; ?></button>
        <script>
            (function(){
                const cells = document.querySelectorAll('[data-cell]');
                const status = document.getElementById('krydsbolle-status');
                const resetBtn = document.getElementById('krydsbolle-reset');

                const config = {
                    start: <?php echo json_encode($start_player); ?>,
                    texts: <?php echo json_encode($texts[$language]); ?>
                };

                let player = (config.start === 'random') ? (Math.random() < 0.5 ? 'X' : 'O') : config.start;
                let board = Array(9).fill(null);
                let active = true;

                function updateStatus() {
                    status.textContent = config.texts.turn + player;
                }

                function checkWinnerAndHighlight() {
                    const combos = [[0,1,2],[3,4,5],[6,7,8],[0,3,6],[1,4,7],[2,5,8],[0,4,8],[2,4,6]];
                    for (const [a,b,c] of combos) {
                        if (board[a] && board[a] === board[b] && board[a] === board[c]) {
                            document.querySelector(`[data-cell="${a}"]`).classList.add('vinder');
                            document.querySelector(`[data-cell="${b}"]`).classList.add('vinder');
                            document.querySelector(`[data-cell="${c}"]`).classList.add('vinder');
                            return board[a];
                        }
                    }
                    return board.every(Boolean) ? 'Draw' : null;
                }

                function resetGame() {
                    board = Array(9).fill(null);
                    player = (config.start === 'random') ? (Math.random() < 0.5 ? 'X' : 'O') : config.start;
                    active = true;
                    cells.forEach(cell => {
                        cell.textContent = '';
                        cell.classList.remove('vinder');
                    });
                    updateStatus();
                }

                cells.forEach(cell => {
                    cell.addEventListener('click', () => {
                        const index = +cell.dataset.cell;
                        if (!board[index] && active) {
                            board[index] = player;
                            cell.textContent = player;
                            const winner = checkWinnerAndHighlight();
                            if (winner) {
                                status.textContent = winner === 'Draw' ? config.texts.draw : winner + config.texts.wins;
                                active = false;
                            } else {
                                player = player === 'X' ? 'O' : 'X';
                                updateStatus();
                            }
                        }
                    });
                });

                resetBtn.addEventListener('click', resetGame);

                // Initial status update
                updateStatus();
            })();
        </script>
    </div>
    <?php
    return ob_get_clean();
}
add_shortcode('tic_tac_toe', 'kjb_vis_spil');
